/* 
 * 	Lifted and modified from JGraphT's JGraphModelAdapter
 */
package easik.sketch.util;
import java.awt.Font;
import java.awt.geom.Rectangle2D;
import java.io.Serializable;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import javax.swing.BorderFactory;


import org.jgraph.event.GraphModelEvent;
import org.jgraph.event.GraphModelListener;
import org.jgraph.event.GraphModelEvent.GraphModelChange;
import org.jgraph.graph.AttributeMap;
import org.jgraph.graph.ConnectionSet;
import org.jgraph.graph.DefaultEdge;
import org.jgraph.graph.DefaultGraphCell;
import org.jgraph.graph.DefaultGraphModel;
import org.jgraph.graph.DefaultPort;
import org.jgraph.graph.Edge;
import org.jgraph.graph.GraphCell;
import org.jgraph.graph.GraphConstants;
import org.jgraph.graph.Port;
import org.jgrapht.event.GraphEdgeChangeEvent;
import org.jgrapht.event.GraphListener;
import org.jgrapht.event.GraphVertexChangeEvent;
import org.jgrapht.graph.ListenableDirectedGraph;

import easik.Easik;
import easik.sketch.Sketch;
import easik.sketch.constraint.Constraint;
import easik.sketch.edge.GuideEdge;
import easik.sketch.edge.SketchEdge;
import easik.sketch.util.graph.EntityCell;
import easik.sketch.vertex.EntityNode;
import easik.sketch.vertex.SketchVertex;

/**
 * This class is the adapter which allows JGraphT graphs to be represented in JGraph
 * I stole this from JGraphT, but modified it into something appropriate for JEase.
 * 
 * @author Rob Fletcher 2005
 * @author Kevin Green 2006
 * @version 2006-08-01 Kevin Green
 */
public class SketchAdapter extends DefaultGraphModel {
	/**
	 * JGraph edges being added
	 */
	private final Set<Edge> m_jEdgesBeingAdded = new HashSet<Edge>();
	/**
	 * JGraph edges being removed
	 */
	private final Set<Edge> m_jEdgesBeingRemoved = new HashSet<Edge>();
	/**
	 * JGraph vertices being added
	 */
	private final Set<DefaultGraphCell> m_jVerticesBeingAdded = new HashSet<DefaultGraphCell>();
	/**
	 * JGraph vertices being removed
	 */
	private final Set<DefaultGraphCell> m_jVerticesBeingRemoved = new HashSet<DefaultGraphCell>();
	/**
	 * JGraphT edges being added
	 */
	private final Set<org.jgrapht.graph.DefaultEdge> m_jtEdgesBeingAdded = new HashSet<org.jgrapht.graph.DefaultEdge>();
	/**
	 * JGraphT edges being removed
	 */
	private final Set<org.jgrapht.graph.DefaultEdge> m_jtEdgesBeingRemoved = new HashSet<org.jgrapht.graph.DefaultEdge>();
	/**
	 * JGraphT vertices being removed
	 */
	private final Set<SketchVertex> m_jtVerticesBeingAdded = new HashSet<SketchVertex>();
	/**
	 * JGraphT vertices being removed
	 */
	private final Set<SketchVertex> m_jtVerticesBeingRemoved = new HashSet<SketchVertex>();
	/**
	 * The default attributes for a simple edge
	 */
	private final AttributeMap m_simpleEdgeAttributes;
	/**
	 * The default attributes for a fletched edge
	 */
	private final AttributeMap m_fletchedEdgeAttributes;
	/**
	 * The default attributes for a virtual edge
	 */
	private final AttributeMap m_virtualEdgeAttributes;
	/**
	 * The default attributes for a virtual vertex
	 */
	private final AttributeMap m_virtualVertexAttributes;
	/**
	 * The default attributes for a default vertex
	 */
	private final AttributeMap m_defaultVertexAttributes;

	/**
	 * The cell factory
	 */
	private CellFactory m_cellFactory;
	/**
	 * The JGraphT graph
	 */
	private ListenableDirectedGraph<SketchVertex, org.jgrapht.graph.DefaultEdge> m_jtGraph;
	/**
	 * The JGraphT graph listener
	 */
	private GraphListener m_jtGraphListener;
	/**
	 * Mapping from cells to edges
	 */
	private Map<GraphCell, org.jgrapht.graph.DefaultEdge> m_cellToEdge = new HashMap<GraphCell, org.jgrapht.graph.DefaultEdge>();
	/**
	 * Mapping from cells to vertices
	 */
	private Map<GraphCell, SketchVertex> m_cellToVertex = new HashMap<GraphCell, SketchVertex>();
	/**
	 * Mapping from edges to cells
	 */
	private Map<org.jgrapht.graph.DefaultEdge, GraphCell> m_edgeToCell = new HashMap<org.jgrapht.graph.DefaultEdge, GraphCell>();
	/**
	 * Mapping from vertices to cells
	 */
	private Map<SketchVertex, GraphCell> m_vertexToCell = new HashMap<SketchVertex, GraphCell>();
	/**
	 * The sketch
	 */
	private Sketch _ourSketch;

	/**
	 * Constructs a new JGraph model adapter for the specified JGraphT graph.	
	 * @param jGraphTGraph the JGraphT graph for which JGraph model adapter to
	 *        be created.
	 * @param inSketch The sketch
	 */
	public SketchAdapter(ListenableDirectedGraph<SketchVertex, org.jgrapht.graph.DefaultEdge> jGraphTGraph, Sketch inSketch) {
		super();
		_ourSketch = inSketch;
		
		m_jtGraph = jGraphTGraph;
		m_defaultVertexAttributes = createDefaultVertexAttributes();
		m_virtualVertexAttributes = createVirtualVertexAttributes();
		m_virtualEdgeAttributes = createVirtualEdgeAttributes();
		m_simpleEdgeAttributes = createSimpleEdgeAttributes();
		m_fletchedEdgeAttributes = createFletchedEdgeAttributes();

		m_cellFactory = new DefaultCellFactory();

		m_jtGraphListener = new JGraphTListener();
		jGraphTGraph.addGraphListener(m_jtGraphListener);

		this.addGraphModelListener(new JGraphListener(_ourSketch));

		for (Iterator i = jGraphTGraph.vertexSet().iterator(); i.hasNext();) {
			addJGraphTVertex(i.next());
		}

		for (Iterator i = jGraphTGraph.edgeSet().iterator(); i.hasNext();) {
			addJGraphTEdge((org.jgrapht.graph.DefaultEdge) i.next());
		}
	}

	/**
	 * For getting the JGraphT edge from a JGraph cell
	 * @param inCell The JGraph cell
	 * @return The JGraphT edge from a JGraph cell
	 */
	public Object getEdgeFromCell(DefaultEdge inCell) {
		if (m_cellToEdge.containsKey(inCell)) {
			return m_cellToEdge.get(inCell);
		} else {
			return null;
		}
	}

	/**
	 * For getting the JGraphT vertex from a JGraph cell
	 * @param inCell The JGraph cell
	 * @return The JGraphT vertex from a JGraph cell
	 */
	public Object getVertexFromCell(DefaultGraphCell inCell) {
		if (m_cellToVertex.containsKey(inCell)) {
			return m_cellToVertex.get(inCell);
		} else {
			return null;
		}
	}

	/** 
	 *	This is method is called whenever a vertex is moved on screen. It updates the saved
	 * of the cell, keeping the jgrapht objects and jgraph cells synchronised
	 * @param inCell
	 */
	public void updateVertexPos(DefaultGraphCell inCell) {
		SketchVertex temp = (SketchVertex)getVertexFromCell(inCell);
		Rectangle2D rect = GraphConstants.getBounds(inCell.getAttributes());		
		temp.setX( (int)rect.getX());
		temp.setY( (int)rect.getY()); 	
	}

	/**
	 * Returns the JGraph edge cell that corresponds to the specified JGraphT
	 * edge. If no corresponding cell found, returns <code>null</code>.
	 *
	 * @param jGraphTEdge a JGraphT edge of the JGraphT graph.
	 *
	 * @return the JGraph edge cell that corresponds to the specified JGraphT
	 *         edge, or <code>null</code> if no corresponding cell found.
	 */
	public DefaultEdge getEdgeCell(org.jgrapht.graph.DefaultEdge jGraphTEdge) {
		return (DefaultEdge) m_edgeToCell.get(jGraphTEdge);
	}

	/**
	 * Returns the JGraph vertex cell that corresponds to the specified JGraphT
	 * vertex. If no corresponding cell found, returns <code>null</code>.
	 *
	 * @param jGraphTVertex a JGraphT vertex of the JGraphT graph.
	 *
	 * @return the JGraph vertex cell that corresponds to the specified JGraphT
	 *         vertex, or <code>null</code> if no corresponding cell found.
	 */
	public DefaultGraphCell getVertexCell(Object jGraphTVertex) {
		return (DefaultGraphCell) m_vertexToCell.get(jGraphTVertex);
	}

	/**
	 * Returns the JGraph port cell that corresponds to the specified JGraphT
	 * vertex. If no corresponding port found, returns <code>null</code>.
	 *
	 * @param jGraphTVertex a JGraphT vertex of the JGraphT graph.
	 *
	 * @return the JGraph port cell that corresponds to the specified JGraphT
	 *         vertex, or <code>null</code> if no corresponding cell found.
	 */
	public DefaultPort getVertexPort(Object jGraphTVertex) {
		DefaultGraphCell vertexCell = getVertexCell(jGraphTVertex);

		if (vertexCell == null) {
			return null;
		} else {
			return (DefaultPort) vertexCell.getChildAt(0);
		}
	}

	/**
	 * Create a default attribute mapping
	 * @return The attribute mapping
	 */
	public static AttributeMap createSimpleEdgeAttributes() {
		AttributeMap map = new AttributeMap();
		GraphConstants.setLineEnd(map, GraphConstants.ARROW_TECHNICAL);
		GraphConstants.setEndFill(map, true);
		GraphConstants.setEndSize(map, 10);
		GraphConstants.setLabelAlongEdge(map, true);
		GraphConstants.setLineColor(map, Easik.getInstance().getIni().getSIMPLE_EDGE_COLOR());
		GraphConstants.setLineStyle(map, GraphConstants.STYLE_BEZIER);
		return map;
	}

	/**
	 * Create a default attribute mapping
	 * @return The attribute mapping
	 */
	public static AttributeMap createVirtualEdgeAttributes() {
		AttributeMap map = new AttributeMap();
		GraphConstants.setLineEnd(map, GraphConstants.ARROW_NONE);
		GraphConstants.setLineBegin(map, GraphConstants.ARROW_NONE);								
		GraphConstants.setDashPattern(map, new float[] { 3.0f });
		GraphConstants.setLineColor(map, Easik.getInstance().getIni().getVIRTUAL_EDGE_COLOR());
		GraphConstants.setValue(map, "");
		return map;
	}

	/**
	 * Create a default attribute mapping
	 * @return The attribute mapping
	 */
	public static AttributeMap createFletchedEdgeAttributes() {
		AttributeMap map = new AttributeMap();
		GraphConstants.setLineEnd(map, GraphConstants.ARROW_TECHNICAL);
		GraphConstants.setEndFill(map, true);
		GraphConstants.setEndSize(map, 10);
		GraphConstants.setLineBegin(map, GraphConstants.ARROW_DIAMOND);
		GraphConstants.setBeginFill(map, false);
		GraphConstants.setBeginSize(map, 10);
		GraphConstants.setLabelAlongEdge(map, true);
		GraphConstants.setLineColor(map, Easik.getInstance().getIni().getFLETCHED_EDGE_COLOR());
		GraphConstants.setLineStyle(map, GraphConstants.STYLE_BEZIER);
		return map;
	}

	/**
	 * Create a default attribute mapping
	 * @return The attribute mapping
	 */
	public static AttributeMap createDefaultVertexAttributes() {
		AttributeMap map = new AttributeMap();
		GraphConstants.setBounds(map, new Rectangle2D.Double(50, 50, 90, 30));
		GraphConstants.setAutoSize(map, true);
		GraphConstants.setInset(map, 5);
		GraphConstants.setBorder(map, BorderFactory.createLineBorder(Easik.getInstance().getIni().getENTITY_BORDER_COLOR(), 1));
		GraphConstants.setBackground(map, Easik.getInstance().getIni().getENTITY_BG_COLOR());
		GraphConstants.setForeground(map, Easik.getInstance().getIni().getENTITY_FG_COLOR());
		GraphConstants.setFont(map,	GraphConstants.DEFAULTFONT.deriveFont(Font.BOLD, 12));
		GraphConstants.setOpaque(map, true);
		return map;
	}

	/**
	 * Create a default attribute mapping
	 * @return The attribute mapping
	 */
	public static AttributeMap createVirtualVertexAttributes() {
		AttributeMap map = new AttributeMap();
		GraphConstants.setBounds(map, new Rectangle2D.Double(30, 30, 30, 30));
		GraphConstants.setBorder(map, BorderFactory.createLineBorder(Easik.getInstance().getIni().getCONSTRAINT_BORDER_COLOR(), 1));
		GraphConstants.setBackground(map, Easik.getInstance().getIni().getCONSTRAINT_BG_COLOR());
		GraphConstants.setForeground(map, Easik.getInstance().getIni().getCONSTRAINT_FG_COLOR());
		GraphConstants.setFont(map, GraphConstants.DEFAULTFONT.deriveFont(Font.BOLD, 12));
		GraphConstants.setOpaque(map, true);
		return map;
	}

	/**
	 * Applies the specified attributes to the model, as in {@link
	 * org.jgraph.graph.GraphModel#edit(java.util.Map,
	 * org.jgraph.graph.ConnectionSet, org.jgraph.graph.ParentMap,
	 * javax.swing.undo.UndoableEdit[])} method.
	 *
	 * @param attrs the attributes to be applied to the model.
	 */
	public void edit(Map attrs) {
		super.edit(attrs, null, null, null);
	}

	/**
	 * Adds an edge corresponding to the specified JGraph edge to the
	 * underlying JGraphT graph. We try to find out to which vertices the edge
	 * is connected; if we find only one or none at all, we remove it again
	 * from the JGraph graph, because we cannot add such an edge to the
	 * JGraphT graph.
	 * 
	 * <p>
	 * This method is to be called only for edges that have already been added
	 * to the JGraph graph.
	 * </p>
	 *
	 * @param jEdge the JGraph Edge to be added.
	 *
	 * @return true if the edge was successfully added, false otherwise.
	 */
	protected boolean addJGraphEdge(org.jgraph.graph.Edge jEdge) {
		org.jgrapht.graph.DefaultEdge jtEdge;
		Object jSource = getSourceVertex(this, jEdge);
		Object jTarget = getTargetVertex(this, jEdge);

		if (!(m_cellToVertex.containsKey(jSource)
			&& m_cellToVertex.containsKey(jTarget))) {
			// This is a 'dangling edge'. We have to remove it in the
			// JGraph.
			// TOAD: Consider alternatives that will allow dangling edges.
			Object[] eArray = { jEdge };
			m_jEdgesBeingRemoved.add(jEdge);
			super.remove(eArray);

			return false;
		}

		SketchVertex jtSource = (SketchVertex)m_cellToVertex.get(jSource);
		SketchVertex jtTarget = (SketchVertex)m_cellToVertex.get(jTarget);

		//jtEdge = new DirectedEdge(jtSource, jtTarget);
		jtEdge = new org.jgrapht.graph.DefaultEdge();
		
		m_jtEdgesBeingAdded.add(jtEdge);

		boolean result = m_jtGraph.addEdge(jtSource, jtTarget, jtEdge);

		if (result) {
			m_cellToEdge.put(jEdge, jtEdge);
			m_edgeToCell.put(jtEdge, jEdge);

			return true;
		} else {
			// Adding the edge failed. We have to remove it from the
			// JGraph too.
			super.remove(new Object[] { jEdge });

			return false;
		}
	}

	/**
	 * Adds the specified JGraphT edge to be reflected by this graph model. To
	 * be called only for edges that already exist in the JGraphT graph.
	 *
	 * @param jtEdge a JGraphT edge to be reflected by this graph model.
	 */
	protected void addJGraphTEdge(org.jgrapht.graph.DefaultEdge jtEdge) {
		
		DefaultEdge edgeCell = m_cellFactory.createEdgeCell(jtEdge);
		
		m_edgeToCell.put(jtEdge, edgeCell);
		m_cellToEdge.put(edgeCell, jtEdge);

		ConnectionSet cs = new ConnectionSet();
		cs.connect(
			edgeCell,
			getVertexPort(Easik.getInstance().getFrame().getSketch().getGraphData().getEdgeSource(jtEdge)),
			getVertexPort(Easik.getInstance().getFrame().getSketch().getGraphData().getEdgeTarget(jtEdge)));
		Object[] cells = { edgeCell };
		AttributeMap attrs = new AttributeMap();

		// Determine which kind of edge is appropriate for drawing		
		if (jtEdge instanceof SketchEdge) {
			if (((SketchEdge) jtEdge).getInjective()) {
				attrs.put(edgeCell, m_fletchedEdgeAttributes.clone());
			} else {
				attrs.put(edgeCell, m_simpleEdgeAttributes.clone());
			}
		} else if (jtEdge instanceof GuideEdge) {
			AttributeMap myMap = (AttributeMap) m_virtualEdgeAttributes.clone();
			if(((GuideEdge)jtEdge).IsHighlighted()){
				GraphConstants.setLineColor(myMap, Easik.getInstance().getIni().getVIRTUAL_HIGHLIGHTED_EDGE_COLOR());
			}
			else{
				GraphConstants.setLineColor(myMap, Easik.getInstance().getIni().getVIRTUAL_EDGE_COLOR());
			}
			attrs.put(edgeCell, myMap);
		}

		m_jEdgesBeingAdded.add(edgeCell);
		super.insert(cells, attrs, cs, null, null);
	}

	/**
	 * Adds the specified JGraphT vertex to be reflected by this graph model.
	 * To be called only for edges that already exist in the JGraphT graph.
	 *
	 * @param jtVertex a JGraphT vertex to be reflected by this graph model.
	 */
	protected void addJGraphTVertex(Object jtVertex) {
		DefaultGraphCell vertexCell;
		if(jtVertex instanceof EntityNode){
			vertexCell = new EntityCell(jtVertex);
		}
		else{
			vertexCell = new DefaultGraphCell(jtVertex);
		}
		vertexCell.add(new DefaultPort());

		m_vertexToCell.put((SketchVertex)jtVertex, vertexCell);
		m_cellToVertex.put(vertexCell, (SketchVertex)jtVertex);

		Object[] cells = { vertexCell };
		AttributeMap attrs = new AttributeMap();

		if (jtVertex instanceof Constraint) {
			attrs.put(vertexCell, m_virtualVertexAttributes.clone());
		} else {
			attrs.put(vertexCell, m_defaultVertexAttributes.clone());
		}

		m_jVerticesBeingAdded.add(vertexCell);
		super.insert(cells, attrs, null, null, null);
	}

	/**
	 * Add a vertex corresponding to this JGraph vertex to the JGraphT graph.
	 * In JGraph, two vertices with the same user object are in principle
	 * allowed; in JGraphT, this would lead to duplicate vertices, which is
	 * not allowed. So if the vertex exists already, we remove it. This method
	 * is to be called only for vertices that have already been added to the
	 * JGraph graph.
	 *
	 * @param jVertex the JGraph vertex to be added.
	 *
	 * @return true if the vertex was successfully added, false otherwise.
	 */
	protected boolean addJGraphVertex(GraphCell jVertex) {
		Object jtVertex;

		if (jVertex instanceof DefaultGraphCell) {
			jtVertex = ((DefaultGraphCell) jVertex).getUserObject();
		} else {
			jtVertex = jVertex.toString();
		}

		if (m_vertexToCell.containsKey(jtVertex)) {
			// We have to remove the new vertex, because it would lead to
			// duplicate vertices. We can't use removeJGraphTVertex for
			// that, because it would remove the wrong (existing) vertex.
			Object[] cells = { jVertex };
			super.remove(cells);

			return false;
		}

		m_jtVerticesBeingAdded.add((SketchVertex)jtVertex);

		boolean result = m_jtGraph.addVertex((SketchVertex)jtVertex);

		m_cellToVertex.put(jVertex, (SketchVertex)jtVertex);
		m_vertexToCell.put((SketchVertex)jtVertex, jVertex);

		return result;
	}

	/**
	 * Remove the edge corresponding to this JGraph edge from the JGraphT
	 * graph. To be called only for edges that have already been removed from
	 * the JGraph graph.
	 *
	 * @param jEdge the JGraph Edge to be removed. If it is not in
	 *        m_cellsToEdges, it is silently ignored.
	 *
	 * @return true if the edge could successfully be removed, false otherwise.
	 */
	protected boolean removeJGraphEdge(org.jgraph.graph.Edge jEdge) {
		if (!m_cellToEdge.containsKey(jEdge)) {
			return false;
		}

		org.jgrapht.graph.DefaultEdge jtEdge =
			(org.jgrapht.graph.DefaultEdge) m_cellToEdge.get(jEdge);

		m_jtEdgesBeingRemoved.add(jtEdge);

		boolean result = m_jtGraph.removeEdge(jtEdge);

		m_cellToEdge.remove(jEdge);
		m_edgeToCell.remove(jtEdge);

		return result;
	}

	/**
	 * Removes the specified JGraphT edge from being reflected by this graph
	 * model. To be called only for edges that have already been removed from
	 * the JGraphT graph.
	 *
	 * @param jtEdge a JGraphT edge to be removed from being reflected by this
	 *        graph model.
	 */
	protected void removeJGraphTEdge(org.jgrapht.graph.DefaultEdge jtEdge) {
		DefaultEdge edgeCell = (DefaultEdge) m_edgeToCell.remove(jtEdge);
		m_cellToEdge.remove(edgeCell);

		Object[] cells = { edgeCell };
		m_jEdgesBeingRemoved.add(edgeCell);
		super.remove(cells);
	}

	/**
	 * Removes the specified JGraphT vertex from being reflected by this graph
	 * model. To be called only for vertices that have already been removed
	 * from the JGraphT graph.
	 *
	 * @param jtVertex a JGraphT vertex to be removed from being reflected by
	 *        this graph model.
	 */
	protected void removeJGraphTVertex(Object jtVertex) {
		DefaultGraphCell vertexCell =
			(DefaultGraphCell) m_vertexToCell.remove(jtVertex);
		m_cellToVertex.remove(vertexCell);

		Object[] cells = { vertexCell, vertexCell.getChildAt(0)};
		m_jVerticesBeingRemoved.add(vertexCell);
		super.remove(cells);
	}

	/**
	 * Remove the vertex corresponding to this JGraph vertex from the JGraphT
	 * graph. If any edges are incident with this vertex, we remove them from
	 * both graphs first, because otherwise the JGraph graph would leave them
	 * intact and the JGraphT graph would throw them out. This method is to be
	 * called only for vertices that have already been removed from the JGraph
	 * graph.
	 *
	 * @param jVertex the JGraph vertex to be removed. If it is not in
	 *        m_cellsToVertices, it is silently ignored.
	 *
	 * @return true if the vertex could successfully be removed, false
	 *         otherwise.
	 */
	protected boolean removeJGraphVertex(GraphCell jVertex) {
		if (!m_cellToVertex.containsKey(jVertex)) {
			return false;
		}

		SketchVertex jtVertex = (SketchVertex) m_cellToVertex.get(jVertex);
		Set incidentEdges = m_jtGraph.edgesOf(jtVertex);

		if (incidentEdges != null) {
			// We can't just call removeAllEdges with this list: that
			// would throw a ConcurrentModificationException. So we create
			// a shallow copy.
			Iterator iterator = incidentEdges.iterator();
			incidentEdges = new HashSet();

			while (iterator.hasNext()) {
				incidentEdges.add(iterator.next());
			}

			m_jtGraph.removeAllEdges(incidentEdges);

			// This also triggers removal of the corresponding JGraph
			// edges.
		}

		m_jtVerticesBeingRemoved.add((SketchVertex)jtVertex);

		boolean result = m_jtGraph.removeVertex(jtVertex);

		m_cellToVertex.remove(jVertex);
		m_vertexToCell.remove(jtVertex);

		return result;
	}

	/**
	 * Creates the JGraph cells that reflect the respective JGraphT elements.
	 *
	 * @author Barak Naveh
	 *
	 * @since Dec 12, 2003
	 */
	public interface CellFactory {
		/**
		 * Creates an edge cell that contains its respective JGraphT edge.
		 *
		 * @param jGraphTEdge a JGraphT edge to be contained.
		 *
		 * @return an edge cell that contains its respective JGraphT edge.
		 */
		public DefaultEdge createEdgeCell(org.jgrapht.graph.DefaultEdge jGraphTEdge);

		/**
		 * Creates a vertex cell that contains its respective JGraphT vertex.
		 *
		 * @param jGraphTVertex a JGraphT vertex to be contained.
		 *
		 * @return a vertex cell that contains its respective JGraphT vertex.
		 */
		public DefaultGraphCell createVertexCell(Object jGraphTVertex);
	}

	/**
	 * A simple default cell factory.
	 *
	 * @author Barak Naveh
	 * @since Dec 12, 2003
	 */
	public class DefaultCellFactory implements CellFactory, Serializable {		
		
		/**
		 * Creates an edge cell
		 * @param jGraphTEdge The JGraphT edge to make the edge cell from
		 * @return The created edge
		 */
		public DefaultEdge createEdgeCell(org.jgrapht.graph.DefaultEdge jGraphTEdge) {
			return new DefaultEdge(jGraphTEdge);
		}
		
		/**
		 * Creates a vertex cell
		 * @param jGraphTVertex The JGraphT vertex to make the vertex from
		 * @return The created vertex
		 */
		public DefaultGraphCell createVertexCell(Object jGraphTVertex) {
			return new DefaultGraphCell(jGraphTVertex);
		}
	}

	/**
	 * <p>
	 * Inner class listening to the GraphModel. If something is changed in the
	 * GraphModel, this Listener gets notified and propagates the change back
	 * to the JGraphT graph, if it didn't originate there.
	 * </p>
	 * 
	 * <p>
	 * If this change contains changes that would make this an illegal JGraphT
	 * graph, like adding an edge that is incident with only one vertex, the
	 * illegal parts of the change are undone.
	 * </p>
	 */
	private class JGraphListener implements GraphModelListener, Serializable {
		/**
		 * The sketch
		 */
		private Sketch _ourSketch;
		
		/**
		 * Default constructor
		 * 
		 * @param inSketch The sketch
		 */
		public JGraphListener(Sketch inSketch){
			super();
			_ourSketch = inSketch;	
		}
		
		/**
		 * This method is called for all JGraph changes.
		 *
		 * @param e
		 */
		public void graphChanged(GraphModelEvent e) {
			// We first remove edges that have to be removed, then we
			// remove vertices, then we add vertices and finally we add
			// edges. Otherwise, things might go wrong: for example, if we
			// would first remove vertices and then edges, removal of the
			// vertices might induce 'automatic' removal of edges. If we
			// later attempt to re-remove these edges, we get confused.
			
			GraphModelChange change = e.getChange();
			Set jEdges = new HashSet();
			Set JVertices = new HashSet();

			Object[] changedCells = change.getChanged();
			Object[] removedArray = change.getRemoved();
			Object[] insertedArray = change.getInserted();
			
			// If a cell has changed, and was not removed or inserted,
			// then we fire its 'update position' method.
			for(int i = 0; i < changedCells.length; i++) {		
				if(changedCells[i].getClass() == DefaultGraphCell.class || changedCells[i].getClass() == EntityCell.class) {
					boolean newOrDeleted = false;
					if(removedArray != null) {					
						for(int j = 0; j < removedArray.length; j++) {
							if(removedArray[j] == changedCells[i]) {
								newOrDeleted = true;
							} 
						}
					}
					if(insertedArray != null) {
						for(int j = 0; j < insertedArray.length; j++) {
							if(insertedArray[j] == changedCells[i]) {
								newOrDeleted = true;
							} 
						}
					}
					if(!newOrDeleted) {					
						_ourSketch.getAdapter().updateVertexPos( (DefaultGraphCell)changedCells[i] );
					}										
				}
			}
			
			// Process the array of removed items					
			if (removedArray != null) {
				filterEdgesAndVertices(removedArray, jEdges, JVertices);
				for (Iterator i = jEdges.iterator(); i.hasNext();) {
					org.jgraph.graph.Edge jEdge =
						(org.jgraph.graph.Edge) i.next();
					if (m_jEdgesBeingRemoved.contains(jEdge)) {
						m_jEdgesBeingRemoved.remove(jEdge);
					} else {
						removeJGraphEdge(jEdge);
					}
				}

				for (Iterator i = JVertices.iterator(); i.hasNext();) {
					GraphCell jVertex = (GraphCell) i.next();
					if (m_jVerticesBeingRemoved.contains(jVertex)) {
						m_jVerticesBeingRemoved.remove(jVertex);
					} else {
						removeJGraphVertex(jVertex);
					}
				}

				jEdges.clear();
				JVertices.clear();
			}
			
			// Process the array of inserted items
			if (insertedArray != null) {
				filterEdgesAndVertices(insertedArray, jEdges, JVertices);
			
				for (Iterator i = JVertices.iterator(); i.hasNext();) {
					GraphCell jVertex = (GraphCell) i.next();			
					if (m_jVerticesBeingAdded.contains(jVertex)) {
						m_jVerticesBeingAdded.remove(jVertex);
					} else {
						addJGraphVertex(jVertex);
					}
				}

				for (Iterator i = jEdges.iterator(); i.hasNext();) {
					org.jgraph.graph.Edge jEdge =
						(org.jgraph.graph.Edge) i.next();				
					if (m_jEdgesBeingAdded.contains(jEdge)) {
						m_jEdgesBeingAdded.remove(jEdge);
					} else {
						addJGraphEdge(jEdge);
					}
				}
			}			
		}

		/**
		 * Filters an array of JGraph GraphCell objects into edges and
		 * vertices. Ports and 'groups' are thrown away.
		 *
		 * @param allCells Array of cells to be filtered. These are just plain
		 *        Objects in principle, but if they aren't GraphCells we can't
		 *        detect what they are. So we ignore the non-GraphCell
		 *        elements.
		 * @param jEdges Set to which all edges are added.
		 * @param jVertices Set to which all vertices are added.
		 */
		private void filterEdgesAndVertices(
			Object[] allCells,
			Set jEdges,
			Set jVertices) {
			for (int i = 0; i < allCells.length; i++) {
				Object current = allCells[i];

				if (current instanceof org.jgraph.graph.Edge) {
					jEdges.add(current);
				} else if (!(current instanceof Port)) {
					if (current instanceof DefaultGraphCell) {
						DefaultGraphCell graphCell = (DefaultGraphCell) current;

						if (graphCell.isLeaf()
							// Note: do not change the order
							|| // of these conditions; the code uses the short-cutting of ||.
						 (
								graphCell.getFirstChild() instanceof Port)) {
							jVertices.add(current);
						}

						// If a DefaultGraphCell has a Port as a child, it is a
					} else if (current instanceof GraphCell) {
						// If it is not a DefaultGraphCell, it doesn't have
						// children.
						jVertices.add(current);
					}

					// Otherwise, this is neither an Edge nor a GraphCell; then we
					// don't know what to do with it. So ignore.
				}
			}
		}
	}

	/**
	 * A listener on the underlying JGraphT graph. This listener is used to
	 * keep the JGraph model in sync. Whenever one of the event handlers is
	 * called, it first checks whether the change is due to a previous change
	 * in the JGraph model. If it is, then no action is taken.
	 *
	 * @author Barak Naveh
	 * @since Aug 2, 2003
	 */
	private class JGraphTListener implements GraphListener, Serializable {
		
		/**
		 * Action called when an edge is added
		 * @param e The action event
		 */
		public void edgeAdded(GraphEdgeChangeEvent e) {
			org.jgrapht.graph.DefaultEdge jtEdge = (org.jgrapht.graph.DefaultEdge)e.getEdge();

			if (m_jtEdgesBeingAdded.contains(jtEdge)) {
				m_jtEdgesBeingAdded.remove(jtEdge);
			} else {
				addJGraphTEdge(jtEdge);
			}
		}

		/**
		 * Action called when an edge is removed
		 * @param e The action event
		 */
		public void edgeRemoved(GraphEdgeChangeEvent e) {
			org.jgrapht.graph.DefaultEdge jtEdge = (org.jgrapht.graph.DefaultEdge)e.getEdge();

			if (m_jtEdgesBeingRemoved.contains(jtEdge)) {
				m_jtEdgesBeingRemoved.remove(jtEdge);
			} else {
				removeJGraphTEdge(jtEdge);
			}
		}

		/**
		 * Action called when a vertex is added
		 * @param e The action event
		 */
		public void vertexAdded(GraphVertexChangeEvent e) {
			Object jtVertex = e.getVertex();

			if (m_jtVerticesBeingAdded.contains(jtVertex)) {
				m_jtVerticesBeingAdded.remove(jtVertex);
			} else {
				addJGraphTVertex(jtVertex);
			}
		}

		/**
		 * Action called when a vertex is removed
		 * @param e The action event
		 */
		public void vertexRemoved(GraphVertexChangeEvent e) {
			Object jtVertex = e.getVertex();

			if (m_jtVerticesBeingRemoved.contains(jtVertex)) {
				m_jtVerticesBeingRemoved.remove(jtVertex);
			} else {
				removeJGraphTVertex(jtVertex);
			}
		}
	}
}
